<?php
if (!defined('ABSPATH')) { exit; }

class PUFSM_Settings {
    const OPT_KEY = 'pufsm_settings';

    public static function init() {
        add_action('admin_init', array(__CLASS__, 'register'));
    }

    public static function register() {
        register_setting('pufsm_settings_group', self::OPT_KEY, array(
            'type' => 'array',
            'sanitize_callback' => array(__CLASS__, 'sanitize'),
            'default' => self::defaults(),
        ));
    }

    public static function defaults() {
        return array(
            'smtp_host' => '',
            'smtp_port' => 587,
            'smtp_secure' => 'tls', // tls|ssl|none
            'smtp_auth' => 1,
            'smtp_user' => '',
            'smtp_pass' => '',
            'from_email' => '',
            'from_name'  => 'پارس آپتایم',
            'reply_to_use_sender' => 1,
            'rate_limit_seconds' => 20,
            'rate_limit_max_per_hour' => 20,

            // Style options
            'style_mode' => 'theme', // theme|bootstrap|none
            'wrapper_class' => '',
            'input_class' => '',
            'button_class' => '',
            'custom_css' => '',
        );
    }

    public static function get() {
        $val = get_option(self::OPT_KEY, array());
        $out = wp_parse_args(is_array($val) ? $val : array(), self::defaults());

        // If From Email is empty, default to SMTP account if it looks like an email
        if (empty($out['from_email']) && !empty($out['smtp_user']) && strpos($out['smtp_user'], '@') !== false) {
            $out['from_email'] = sanitize_email($out['smtp_user']);
        }

        // Normalize secure
        if (!in_array($out['smtp_secure'], array('tls','ssl','none'), true)) {
            $out['smtp_secure'] = self::defaults()['smtp_secure'];
        }

        // Normalize style mode
        if (!in_array($out['style_mode'], array('theme','bootstrap','none'), true)) {
            $out['style_mode'] = self::defaults()['style_mode'];
        }

        // Normalize classes
        foreach (array('wrapper_class','input_class','button_class') as $k) {
            $out[$k] = is_string($out[$k]) ? trim($out[$k]) : '';
        }
        $out['custom_css'] = is_string($out['custom_css']) ? (string)$out['custom_css'] : '';

        return $out;
    }

    public static function sanitize($input) {
        $d = self::defaults();
        $out = array();

        $out['smtp_host'] = isset($input['smtp_host']) ? sanitize_text_field($input['smtp_host']) : $d['smtp_host'];
        $out['smtp_port'] = isset($input['smtp_port']) ? max(1, (int)$input['smtp_port']) : (int)$d['smtp_port'];

        $secure = isset($input['smtp_secure']) ? sanitize_text_field($input['smtp_secure']) : $d['smtp_secure'];
        $out['smtp_secure'] = in_array($secure, array('tls','ssl','none'), true) ? $secure : $d['smtp_secure'];

        $out['smtp_auth'] = !empty($input['smtp_auth']) ? 1 : 0;
        $out['smtp_user'] = isset($input['smtp_user']) ? sanitize_text_field($input['smtp_user']) : '';
        // keep secret as-is unless provided
        if (!empty($input['smtp_pass'])) {
            $out['smtp_pass'] = $input['smtp_pass'];
        } else {
            $existing = self::get();
            $out['smtp_pass'] = isset($existing['smtp_pass']) ? $existing['smtp_pass'] : '';
        }

        $out['from_email'] = isset($input['from_email']) ? sanitize_email($input['from_email']) : '';
        $out['from_name']  = isset($input['from_name']) ? sanitize_text_field($input['from_name']) : $d['from_name'];

        $out['reply_to_use_sender'] = !empty($input['reply_to_use_sender']) ? 1 : 0;

        
        // Style options
        $mode = isset($input['style_mode']) ? sanitize_text_field($input['style_mode']) : $d['style_mode'];
        $out['style_mode'] = in_array($mode, array('theme','bootstrap','none'), true) ? $mode : $d['style_mode'];

        $out['wrapper_class'] = isset($input['wrapper_class']) ? sanitize_text_field($input['wrapper_class']) : $d['wrapper_class'];
        $out['input_class']   = isset($input['input_class']) ? sanitize_text_field($input['input_class']) : $d['input_class'];
        $out['button_class']  = isset($input['button_class']) ? sanitize_text_field($input['button_class']) : $d['button_class'];

        // Allow CSS (sanitized with kses, still stored as plain text)
        $out['custom_css'] = isset($input['custom_css']) ? wp_kses_post($input['custom_css']) : $d['custom_css'];
$out['rate_limit_seconds'] = isset($input['rate_limit_seconds']) ? max(0, (int)$input['rate_limit_seconds']) : (int)$d['rate_limit_seconds'];
        $out['rate_limit_max_per_hour'] = isset($input['rate_limit_max_per_hour']) ? max(1, (int)$input['rate_limit_max_per_hour']) : (int)$d['rate_limit_max_per_hour'];

        return $out;
    }
}
