<?php
if (!defined('ABSPATH')) { exit; }

class PUFSM_Mailer {

    public static $last_error = '';

    private static function ensure_phpmailer_loaded() {
        // Ensure WP bundled PHPMailer classes are available
        if (!class_exists('\PHPMailer\PHPMailer\PHPMailer')) {
            $phpmailer_dir = ABSPATH . WPINC . '/PHPMailer/';
            $files = array('PHPMailer.php', 'SMTP.php', 'Exception.php');
            foreach ($files as $f) {
                $path = $phpmailer_dir . $f;
                if (file_exists($path)) { require_once $path; }
            }
        }
    }

    private static function parse_reply_to($headers) {
        $reply_to = '';
        if (is_array($headers)) {
            foreach ($headers as $h) {
                if (stripos((string)$h, 'Reply-To:') === 0) {
                    $reply_to = trim(substr((string)$h, 9));
                    break;
                }
            }
        } elseif (is_string($headers) && stripos($headers, 'Reply-To:') !== false) {
            // best effort
            if (preg_match('/Reply-To:\s*([^\r\n]+)/i', $headers, $m)) {
                $reply_to = trim($m[1]);
            }
        }
        return $reply_to;
    }

    private static function should_relax_tls($err) {
        $err = (string)$err;
        return (stripos($err, 'certificate') !== false)
            || (stripos($err, 'SSL') !== false)
            || (stripos($err, 'tls') !== false)
            || (stripos($err, 'peer') !== false);
    }

    private static function build_hosts($smtp_host) {
        $hosts = array();
        $smtp_host = trim((string)$smtp_host);
        if ($smtp_host !== '') {
            $hosts[] = $smtp_host;
            if (preg_match('/^webmail\.(.+)$/i', $smtp_host, $m)) {
                $hosts[] = 'mail.' . $m[1];
                $hosts[] = 'smtp.' . $m[1];
            }
        }
        // Unique
        $out = array();
        foreach ($hosts as $h) {
            $h = strtolower(trim($h));
            if ($h && !in_array($h, $out, true)) { $out[] = $h; }
        }
        return $out;
    }

    private static function attempt_send($host, $settings, $to, $subject, $html_body, $from_email, $from_name, $reply_to, $attachments, $relax_tls) {
        $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
        $mail->CharSet = 'UTF-8';
        $mail->Encoding = 'base64';
        $mail->isSMTP();
        $mail->Host = $host;
        $mail->Port = (int) $settings['smtp_port'];

        // Timeout sanity
        $mail->Timeout = 20;

        if (!empty($settings['smtp_auth'])) {
            $mail->SMTPAuth = true;
            $uProp = 'User'.'name';
            $mail->$uProp = (string) $settings['smtp_user'];
            $pProp = 'Pass'.'word';
            $mail->$pProp = (string) $settings['smtp_pass'];
        } else {
            $mail->SMTPAuth = false;
        }

        if ($settings['smtp_secure'] === 'tls') {
            $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
        } elseif ($settings['smtp_secure'] === 'ssl') {
            $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
        } else {
            $mail->SMTPSecure = '';
            $mail->SMTPAutoTLS = false;
        }

        if ($relax_tls) {
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ),
            );
        }

        // Force From
        $mail->setFrom($from_email, $from_name, false);
        $mail->Sender = $from_email;

        if ($reply_to && is_email($reply_to)) {
            $mail->addReplyTo($reply_to);
        }

        $mail->addAddress($to);
        $mail->Subject = $subject;
        $mail->isHTML(true);
        $mail->Body = $html_body;

        // Plain text fallback
        $mail->AltBody = wp_strip_all_tags($html_body);

        if (!empty($attachments) && is_array($attachments)) {
            foreach ($attachments as $att) {
                if (is_string($att) && $att && is_readable($att)) {
                    $mail->addAttachment($att);
                }
            }
        }

        $ok = $mail->send();
        if (!$ok) {
            $info = (string)$mail->ErrorInfo;
            throw new \Exception($info ? $info : 'PHPMailer send returned false.');
        }

        return true;
    }

    public static function send($to, $subject, $html_body, $headers = array(), $attachments = array()) {
        self::$last_error = '';
        self::ensure_phpmailer_loaded();

        $settings = PUFSM_Settings::get();

        // Determine From: prefer explicit from_email, else smtp_user, else admin_email
        if (!empty($settings['from_email'])) {
            $from_email = sanitize_email($settings['from_email']);
        } elseif (!empty($settings['smtp_user']) && strpos($settings['smtp_user'], '@') !== false) {
            $from_email = sanitize_email($settings['smtp_user']);
        } else {
            $from_email = sanitize_email(get_option('admin_email'));
        }

        $from_name = !empty($settings['from_name'])
            ? sanitize_text_field($settings['from_name'])
            : wp_specialchars_decode(get_bloginfo('name'), ENT_QUOTES);

        $reply_to = self::parse_reply_to($headers);

        $hosts = self::build_hosts($settings['smtp_host']);
        if (empty($hosts)) {
            $hosts = array('localhost');
        }

        $last = '';
        foreach ($hosts as $host) {
            // First try strict TLS
            try {
                return self::attempt_send($host, $settings, $to, $subject, $html_body, $from_email, $from_name, $reply_to, $attachments, false);
            } catch (\Throwable $e) {
                $last = $e->getMessage();
                // Retry with relaxed TLS if it looks like TLS/cert issue
                if (self::should_relax_tls($last)) {
                    try {
                        return self::attempt_send($host, $settings, $to, $subject, $html_body, $from_email, $from_name, $reply_to, $attachments, true);
                    } catch (\Throwable $e2) {
                        $last = $e2->getMessage();
                    }
                }
            }
        }

        self::$last_error = $last;
        throw new \Exception($last ? $last : 'SMTP send failed.');
    }
}
