<?php
if (!defined('ABSPATH')) { exit; }

class PUFSM_Forms {

    private static $last_result = array(); // [form_id => ['type'=>'success|error','message'=>'...']]

    public static function init() {
        add_action('init', array(__CLASS__, 'register_assets'));

        // AJAX submit (preferred: no redirect)
        add_action('wp_ajax_pufsm_submit_form', array(__CLASS__, 'ajax_submit'));
        add_action('wp_ajax_nopriv_pufsm_submit_form', array(__CLASS__, 'ajax_submit'));
    }

    public static function register_assets() {
        wp_register_style('pufsm-frontend', PUFSM_PLUGIN_URL . 'assets/css/frontend.css', array(), PUFSM_VERSION);
        // Custom inline CSS only (no file)
        wp_register_style('pufsm-custom', false, array(), PUFSM_VERSION);
        wp_register_script('pufsm-frontend', PUFSM_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), PUFSM_VERSION, true);

        wp_localize_script('pufsm-frontend', 'PUFSM', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'ajax_nonce' => wp_create_nonce('pufsm_ajax'),
        ));
    }

    public static function get_form($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'pufsm_forms';
        $row = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id=%d", $id), ARRAY_A);
        return $row ? $row : null;
    }

    public static function list_forms() {
        global $wpdb;
        $table = $wpdb->prefix . 'pufsm_forms';
        return $wpdb->get_results("SELECT * FROM $table ORDER BY id DESC", ARRAY_A);
    }

    public static function create_form($data) {
        global $wpdb;
        $table = $wpdb->prefix . 'pufsm_forms';
        $now = current_time('mysql');
        $wpdb->insert($table, array(
            'name' => $data['name'],
            'recipient' => $data['recipient'],
            'subject' => $data['subject'],
            'fields' => $data['fields'],
            'success_message' => $data['success_message'],
            'is_enabled' => (int) $data['is_enabled'],
            'created_at' => $now,
            'updated_at' => $now,
        ), array('%s','%s','%s','%s','%s','%d','%s','%s'));
        return (int) $wpdb->insert_id;
    }

    public static function update_form($id, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'pufsm_forms';
        $now = current_time('mysql');
        return (bool) $wpdb->update($table, array(
            'name' => $data['name'],
            'recipient' => $data['recipient'],
            'subject' => $data['subject'],
            'fields' => $data['fields'],
            'success_message' => $data['success_message'],
            'is_enabled' => (int) $data['is_enabled'],
            'updated_at' => $now,
        ), array('id' => (int)$id), array('%s','%s','%s','%s','%s','%d','%s'), array('%d'));
    }

    public static function delete_form($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'pufsm_forms';
        return (bool) $wpdb->delete($table, array('id' => (int)$id), array('%d'));
    }

    private static function rate_limit_key($form_id) {
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '0.0.0.0';
        return 'pufsm_rl_' . md5($ip . '|' . (int)$form_id);
    }

    private static function is_rate_limited($form_id) {
        $settings = PUFSM_Settings::get();
        $key = self::rate_limit_key($form_id);

        $state = get_transient($key);
        if (!is_array($state)) {
            $state = array('last' => 0, 'count' => 0, 'hour_start' => time());
        }

        $now = time();
        if (($now - (int)$state['hour_start']) >= 3600) {
            $state['count'] = 0;
            $state['hour_start'] = $now;
        }

        if (!empty($settings['rate_limit_seconds']) && (int)$settings['rate_limit_seconds'] > 0) {
            if ((int)$state['last'] > 0 && ($now - (int)$state['last']) < (int)$settings['rate_limit_seconds']) {
                return true;
            }
        }

        if (!empty($settings['rate_limit_max_per_hour']) && (int)$settings['rate_limit_max_per_hour'] > 0) {
            if ((int)$state['count'] >= (int)$settings['rate_limit_max_per_hour']) {
                return true;
            }
        }

        $state['last'] = $now;
        $state['count'] = (int)$state['count'] + 1;
        set_transient($key, $state, 2 * 3600);

        return false;
    }

    private static function to_fa_digits($str) {
        $map = array('0'=>'۰','1'=>'۱','2'=>'۲','3'=>'۳','4'=>'۴','5'=>'۵','6'=>'۶','7'=>'۷','8'=>'۸','9'=>'۹');
        return strtr((string)$str, $map);
    }

    private static function normalize_digits($str) {
        $map = array(
            '۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4','۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9',
            '٠'=>'0','١'=>'1','٢'=>'2','٣'=>'3','٤'=>'4','٥'=>'5','٦'=>'6','٧'=>'7','٨'=>'8','٩'=>'9',
        );
        $s = strtr((string)$str, $map);
        $s = preg_replace('/[^\d\-]/u', '', $s);
        return $s;
    }

    private static function captcha_make($form_id) {
        // Generate a slightly harder math captcha (Persian words, + / - / ×)
        $a = wp_rand(1, 9);
        $b = wp_rand(1, 9);

        // Weighted ops: +, -, *
        $ops = array('+', '-', '*');
        $op = $ops[wp_rand(0, 2)];

        // Keep subtraction non-negative for usability
        if ($op === '-' && $b > $a) { $tmp=$a; $a=$b; $b=$tmp; }

        // Keep multiplication small (1..9)
        if ($op === '*') {
            $a = wp_rand(2, 9);
            $b = wp_rand(2, 9);
        }

        $ts = time();
        $token = hash_hmac('sha256', (int)$form_id . '|' . $a . '|' . $op . '|' . $b . '|' . $ts, wp_salt('pufsm'));

        $op_text = '+';
        if ($op === '+') {
            $op_texts = array('بعلاوه', 'به‌علاوه');
            $op_text = $op_texts[wp_rand(0, count($op_texts) - 1)];
        } elseif ($op === '-') {
            $op_text = 'منهای';
        } else { // *
            $op_texts = array('ضربدر', '×');
            $op_text = $op_texts[wp_rand(0, count($op_texts) - 1)];
        }

        $question = self::to_fa_digits($a) . ' ' . $op_text . ' ' . self::to_fa_digits($b);

        return array(
            'a' => $a,
            'b' => $b,
            'op' => $op,
            'ts' => $ts,
            'token' => $token,
            'question' => $question,
        );
    }

    private static function captcha_verify($form_id, $a, $op, $b, $ts, $token, $user_answer) {
        $form_id = (int)$form_id;
        $a = (int)$a;
        $b = (int)$b;
        $op = ($op === '-') ? '-' : '+';
        $ts = (int)$ts;
        $token = sanitize_text_field((string)$token);

        if ($ts <= 0 || (time() - $ts) > 10 * MINUTE_IN_SECONDS) {
            return false;
        }

        $expected = hash_hmac('sha256', $form_id . '|' . $a . '|' . $op . '|' . $b . '|' . $ts, wp_salt('pufsm'));
        if (!hash_equals($expected, $token)) {
            return false;
        }

        $ans = 0;
        if ($op === '+') { $ans = ($a + $b); }
        elseif ($op === '-') { $ans = ($a - $b); }
        elseif ($op === '*') { $ans = ($a * $b); }
        else { return false; }
        $normalized = self::normalize_digits($user_answer);
        if ($normalized === '' || !preg_match('/^-?\d+$/', $normalized)) { return false; }

        return ((int)$normalized === (int)$ans);
    }

    public static function log_send($form_id, $to, $subject, $status, $error_text = '', $sender_email = '', $payload = '') {
        global $wpdb;
        $table = $wpdb->prefix . 'pufsm_logs';

        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $ua = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_textarea_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $url = (is_ssl() ? 'https://' : 'http://') . (isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '') . (isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '');

        $wpdb->insert($table, array(
            'form_id' => (int)$form_id,
            'to_email' => (string)$to,
            'subject' => (string)$subject,
            'status' => $status ? 1 : 0,
            'error_text' => $error_text,
            'sender_email' => $sender_email,
            'payload' => $payload,
            'ip' => $ip,
            'user_agent' => $ua,
            'page_url' => $url,
            'created_at' => current_time('mysql'),
        ), array('%d','%s','%s','%d','%s','%s','%s','%s','%s','%s','%s'));
    }

    private static function process_submission($form_id, $raw_post) {
        $form_id = (int)$form_id;
        $form = self::get_form($form_id);
        if (!$form || empty($form['is_enabled'])) {
            return array('ok'=>false, 'message'=>'این فرم فعال نیست.', 'code'=>'disabled');
        }

        // honeypot
        if (!empty($raw_post['pufsm_hp'])) {
            return array('ok'=>false, 'message'=>'درخواست نامعتبر است.', 'code'=>'hp');
        }

        // nonce (per-form or AJAX)
        $nonce = isset($raw_post['pufsm_nonce']) ? (string)$raw_post['pufsm_nonce'] : '';
        $nonce_ok = false;

        if ($nonce) {
            $nonce_ok = wp_verify_nonce(sanitize_text_field($nonce), 'pufsm_submit_' . $form_id);
        } else {
            // Fallback: if AJAX nonce exists (AJAX endpoint already validates it), accept here too
            $ajax_nonce = isset($raw_post['ajax_nonce']) ? (string)$raw_post['ajax_nonce'] : '';
            if ($ajax_nonce && wp_verify_nonce(sanitize_text_field($ajax_nonce), 'pufsm_ajax')) {
                $nonce_ok = true;
            }
        }

        if (!$nonce_ok) {
            return array('ok'=>false, 'message'=>'توکن امنیتی نامعتبر است. صفحه را رفرش کنید.', 'code'=>'nonce');
        }

// captcha
        $cap_a = isset($raw_post['pufsm_cap_a']) ? $raw_post['pufsm_cap_a'] : '';
        $cap_b = isset($raw_post['pufsm_cap_b']) ? $raw_post['pufsm_cap_b'] : '';
        $cap_op = isset($raw_post['pufsm_cap_op']) ? $raw_post['pufsm_cap_op'] : '';
        $cap_ts = isset($raw_post['pufsm_cap_ts']) ? $raw_post['pufsm_cap_ts'] : '';
        $cap_token = isset($raw_post['pufsm_cap_token']) ? $raw_post['pufsm_cap_token'] : '';
        $cap_ans = isset($raw_post['pufsm_captcha']) ? $raw_post['pufsm_captcha'] : '';

        if (!self::captcha_verify($form_id, $cap_a, $cap_op, $cap_b, $cap_ts, $cap_token, $cap_ans)) {
            return array('ok'=>false, 'message'=>'کپچا صحیح نیست. لطفاً دوباره تلاش کنید.', 'code'=>'captcha');
        }

        if (self::is_rate_limited($form_id)) {
            return array('ok'=>false, 'message'=>'ارسال بیش از حد مجاز است. لطفاً چند لحظه بعد دوباره تلاش کنید.', 'code'=>'rate');
        }

        $fields = json_decode((string)$form['fields'], true);
        if (!is_array($fields)) { $fields = array(); }

        $values = array();
        $errors = array();
        $sender_email = '';

        $posted_fields = isset($raw_post['pufsm']) && is_array($raw_post['pufsm']) ? $raw_post['pufsm'] : array();

        foreach ($fields as $f) {
            $name = isset($f['name']) ? sanitize_key($f['name']) : '';
            if (!$name) { continue; }

            $type = isset($f['type']) ? sanitize_key($f['type']) : 'text';
            $required = !empty($f['required']);

            $raw = isset($posted_fields[$name]) ? $posted_fields[$name] : '';
            if (is_array($raw)) {
                $val = array_map('sanitize_text_field', array_map('wp_unslash', $raw));
                $val_display = implode(', ', $val);
            } else {
                $raw = wp_unslash($raw);
                if ($type === 'email') {
                    $val = sanitize_email($raw);
                } elseif ($type === 'textarea') {
                    $val = sanitize_textarea_field($raw);
                } else {
                    $val = sanitize_text_field($raw);
                }
                $val_display = $val;
            }

            if ($required && (string)$val_display === '') {
                $label = isset($f['label']) ? sanitize_text_field($f['label']) : $name;
                $errors[] = 'فیلد «' . $label . '» الزامی است.';
            }

            if ($type === 'email' && (string)$val_display !== '' && !is_email($val_display)) {
                $label = isset($f['label']) ? sanitize_text_field($f['label']) : $name;
                $errors[] = 'فرمت ایمیل در فیلد «' . $label . '» صحیح نیست.';
            }

            if (!$sender_email && $type === 'email' && is_email($val_display)) {
                $sender_email = $val_display;
            }

            $values[$name] = array(
                'label' => isset($f['label']) ? sanitize_text_field($f['label']) : $name,
                'value' => $val_display,
                'type'  => $type,
            );
        }

        if (!empty($errors)) {
            return array('ok'=>false, 'message'=>implode('<br>', array_map('esc_html', $errors)), 'code'=>'validation');
        }

        // Subject with tokens {field}
        $subject = (string) $form['subject'];
        foreach ($values as $k => $v) {
            $subject = str_replace('{' . $k . '}', (string)$v['value'], $subject);
        }
        $subject = wp_strip_all_tags($subject);

        $payload = wp_json_encode(array(
            'fields' => $values,
            'sender_email' => $sender_email,
        ), JSON_UNESCAPED_UNICODE);

        // Email body
        $site_name = wp_specialchars_decode(get_bloginfo('name'), ENT_QUOTES);
        $page_url = (is_ssl() ? 'https://' : 'http://') . (isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '') . (isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '');

        $rows_html = '';
        foreach ($values as $v) {
            $rows_html .= '<tr><th style="text-align:right;padding:8px;border:1px solid #e5e7eb;background:#f9fafb;white-space:nowrap;">'
                . esc_html($v['label']) . '</th><td style="text-align:right;padding:8px;border:1px solid #e5e7eb;">'
                . nl2br(esc_html((string)$v['value'])) . '</td></tr>';
        }

        $html = '<div dir="rtl" style="font-family:Tahoma,Arial,sans-serif;font-size:14px;line-height:1.8;color:#111;">'
              . '<h2 style="margin:0 0 12px 0;font-size:18px;">پیام جدید از فرم سایت</h2>'
              . '<p style="margin:0 0 12px 0;">سایت: <strong>' . esc_html($site_name) . '</strong></p>'
              . '<table cellspacing="0" cellpadding="0" style="border-collapse:collapse;width:100%;max-width:760px;">'
              . $rows_html
              . '</table>'
              . '<p style="margin:12px 0 0 0;color:#555;font-size:12px;">آدرس صفحه: ' . esc_html($page_url) . '</p>'
              . '</div>';

        // Recipient
        $to = sanitize_email($form['recipient']);
        if (!$to || !is_email($to)) {
            self::log_send($form_id, (string)$to, $subject, 0, 'Invalid recipient.', $sender_email, $payload);
            return array('ok'=>false, 'message'=>'ایمیل گیرنده معتبر نیست. Recipient را در تنظیمات فرم اصلاح کنید.', 'code'=>'recipient');
        }

        $headers = array();
        $settings = PUFSM_Settings::get();
        if (!empty($settings['reply_to_use_sender']) && $sender_email) {
            $headers[] = 'Reply-To: ' . $sender_email;
        }

        $error_text = '';
        $sent = false;

        try {
            $sent = PUFSM_Mailer::send($to, $subject, $html, $headers);
            if (!$sent) { $error_text = 'send returned false.'; }
        } catch (\Throwable $e) {
            $sent = false;
            $error_text = $e->getMessage();
        }

        self::log_send($form_id, $to, $subject, $sent ? 1 : 0, $error_text, $sender_email, $payload);

        if ($sent) {
            return array('ok'=>true, 'message'=>(string)$form['success_message'], 'code'=>'sent');
        }

        $msg = 'ارسال ایمیل ناموفق بود. لطفاً تنظیمات SMTP را بررسی کنید.';
        if (current_user_can('manage_options') && !empty($error_text)) {
            $msg .= ' ' . '<br><small><code dir="ltr">' . esc_html($error_text) . '</code></small>';
        }
        return array('ok'=>false, 'message'=>$msg, 'code'=>'send');
    }

    public static function ajax_submit() {
        // Basic nonce for AJAX endpoint (separate from per-form nonce)
        $ajax_nonce = isset($_POST['ajax_nonce']) ? sanitize_text_field(wp_unslash($_POST['ajax_nonce'])) : '';
        if (!$ajax_nonce || !wp_verify_nonce($ajax_nonce, 'pufsm_ajax')) {
            wp_send_json(array('ok'=>false, 'message'=>'درخواست نامعتبر است.', 'code'=>'ajax_nonce'), 403);
        }

        $form_id = isset($_POST['pufsm_form_id']) ? (int) $_POST['pufsm_form_id'] : 0;
        if (!$form_id) {
            wp_send_json(array('ok'=>false, 'message'=>'شناسه فرم نامعتبر است.', 'code'=>'form_id'), 400);
        }

        // Build a raw_post-like array (already in $_POST)
        $result = self::process_submission($form_id, $_POST);

        wp_send_json($result, $result['ok'] ? 200 : 400);
    }

    public static function render_form($form_id) {
        $form_id = (int)$form_id;
        $form = self::get_form($form_id);
        if (!$form || empty($form['is_enabled'])) {
            return '<div class="pufsm-note pufsm-error" dir="rtl">این فرم فعال نیست.</div>';
        }

        $settings = PUFSM_Settings::get();
        $mode = isset($settings['style_mode']) ? $settings['style_mode'] : 'theme';

        // Classes
        $wrapper_extra = !empty($settings['wrapper_class']) ? trim((string)$settings['wrapper_class']) : '';
        $input_extra   = !empty($settings['input_class']) ? trim((string)$settings['input_class']) : '';
        $button_extra  = !empty($settings['button_class']) ? trim((string)$settings['button_class']) : '';

        $wrap_class = 'pufsm-wrap' . ($wrapper_extra ? (' ' . $wrapper_extra) : '');
        $form_class = 'pufsm-form';
        $field_wrap_class = 'pufsm-field';
        $label_class = 'pufsm-label';

        $note_success_class = 'pufsm-note pufsm-success';
        $note_error_class   = 'pufsm-note pufsm-error';

        $input_class = $input_extra;
        $select_class = $input_extra;
        $textarea_class = $input_extra;

        $btn_class = $button_extra ? $button_extra : 'pufsm-btn';

        $captcha_row_class = 'pufsm-captcha-row';
        $captcha_q_class   = 'pufsm-captcha-q';

        // Backward-compatible defaults for theme mode (keeps current styling OK)
        if ($mode === 'theme') {
            $wrap_class .= ' apar-form-wrapper';
            $form_class .= ' apar-form';
            if (!$button_extra) {
                $btn_class = 'apar-btn apar-btn--primary pufsm-btn';
            }
        }

        // Bootstrap mode
        if ($mode === 'bootstrap') {
            $wrap_class .= ' pufsm-bs';
            $label_class = 'form-label';
            $field_wrap_class = 'mb-3';

            $note_success_class = 'alert alert-success';
            $note_error_class   = 'alert alert-danger';

            $input_class = $input_extra ? $input_extra : 'form-control';
            $textarea_class = $input_class;
            $select_class = $input_extra ? $input_extra : 'form-select';

            $btn_class = $button_extra ? $button_extra : 'btn btn-primary';

            $captcha_row_class = 'input-group';
            $captcha_q_class   = 'input-group-text';
        }

        // Styles: theme/bootstrap load the plugin CSS; none mode does not.
        if ($mode !== 'none') {
            wp_enqueue_style('pufsm-frontend');
        }

        // Custom CSS: always load if provided
        if (!empty($settings['custom_css'])) {
            wp_enqueue_style('pufsm-custom');
            wp_add_inline_style('pufsm-custom', (string)$settings['custom_css']);
        }

        // Script (AJAX submit + in-page success handling)
        wp_enqueue_script('pufsm-frontend');
        wp_localize_script('pufsm-frontend', 'PUFSM', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'mode' => $mode,
            'note_success_class' => $note_success_class,
            'note_error_class' => $note_error_class,
        ));

        // If this page received a POST for this form (no JS), process and show message (no redirect)
        if (!empty($_POST['pufsm_submit']) && !empty($_POST['pufsm_form_id']) && (int)$_POST['pufsm_form_id'] === $form_id) {
            $res = self::process_submission($form_id, $_POST);
            self::$last_result[$form_id] = array(
                'type' => $res['ok'] ? 'success' : 'error',
                'message' => $res['message'],
                'ok' => $res['ok'],
            );
        }

        // If success, show message instead of form
        if (!empty(self::$last_result[$form_id]) && !empty(self::$last_result[$form_id]['ok'])) {
            $msg = self::$last_result[$form_id]['message'];
            return '<div class="' . esc_attr($wrap_class) . '" id="pufsm-' . esc_attr($form_id) . '" dir="rtl">'
                 . '<div class="' . esc_attr($note_success_class) . '">' . wp_kses_post($msg) . '</div>'
                 . '</div>';
        }

        $flash = !empty(self::$last_result[$form_id]) ? self::$last_result[$form_id] : null;

        $fields = json_decode((string)$form['fields'], true);
        if (!is_array($fields)) { $fields = array(); }

        $captcha = self::captcha_make($form_id);

        ob_start();
        ?>
        <div class="<?php echo esc_attr($wrap_class); ?>" id="pufsm-<?php echo esc_attr($form_id); ?>" dir="rtl">
            <?php if ($flash && !empty($flash['message'])): ?>
                <div class="<?php echo esc_attr(($flash['type'] === 'success') ? $note_success_class : $note_error_class); ?>">
                    <?php echo wp_kses_post($flash['message']); ?>
                </div>
            <?php endif; ?>

            <form class="<?php echo esc_attr($form_class); ?>" method="post" action="" data-pufsm-form="<?php echo esc_attr($form_id); ?>">
                <input type="hidden" name="pufsm_form_id" value="<?php echo esc_attr($form_id); ?>">
                <input type="hidden" name="pufsm_submit" value="1">
                <input type="hidden" name="ajax_nonce" value="<?php echo esc_attr(wp_create_nonce('pufsm_ajax')); ?>">
                <input type="hidden" name="pufsm_nonce" value="<?php echo esc_attr(wp_create_nonce('pufsm_submit_' . $form_id)); ?>">
                <input type="text" name="pufsm_hp" value="" class="pufsm-hp" tabindex="-1" autocomplete="off" aria-hidden="true">

                <?php foreach ($fields as $f):
                    $name = isset($f['name']) ? sanitize_key($f['name']) : '';
                    if (!$name) { continue; }
                    $label = isset($f['label']) ? sanitize_text_field($f['label']) : $name;
                    $type  = isset($f['type']) ? sanitize_key($f['type']) : 'text';
                    $required = !empty($f['required']);
                    $placeholder = isset($f['placeholder']) ? sanitize_text_field($f['placeholder']) : '';
                    $options = isset($f['options']) && is_array($f['options']) ? $f['options'] : array();
                    ?>
                    <div class="<?php echo esc_attr($field_wrap_class); ?>">
                        <label class="<?php echo esc_attr($label_class); ?>" for="pufsm-<?php echo esc_attr($name); ?>">
                            <?php echo esc_html($label); ?>
                            <?php if ($required): ?><span class="pufsm-req">*</span><?php endif; ?>
                        </label>

                        <?php if ($type === 'textarea'): ?>
                            <textarea id="pufsm-<?php echo esc_attr($name); ?>" name="pufsm[<?php echo esc_attr($name); ?>]" rows="5" <?php echo $required ? 'required' : ''; ?> placeholder="<?php echo esc_attr($placeholder); ?>" class="<?php echo esc_attr($textarea_class); ?>"></textarea>
                        <?php elseif ($type === 'select'): ?>
                            <select id="pufsm-<?php echo esc_attr($name); ?>" name="pufsm[<?php echo esc_attr($name); ?>]" <?php echo $required ? 'required' : ''; ?> class="<?php echo esc_attr($select_class); ?>">
                                <option value="">انتخاب کنید</option>
                                <?php foreach ($options as $opt):
                                    $opt = is_string($opt) ? $opt : '';
                                    ?>
                                    <option value="<?php echo esc_attr($opt); ?>"><?php echo esc_html($opt); ?></option>
                                <?php endforeach; ?>
                            </select>
                        <?php elseif ($type === 'checkbox'): ?>
                            <?php if ($mode === 'bootstrap'): ?>
                                <div class="form-check">
                                    <input id="pufsm-<?php echo esc_attr($name); ?>" class="form-check-input <?php echo esc_attr($input_class); ?>" type="checkbox" name="pufsm[<?php echo esc_attr($name); ?>]" value="1">
                                    <label class="form-check-label" for="pufsm-<?php echo esc_attr($name); ?>">
                                        <?php echo esc_html($placeholder ? $placeholder : 'تأیید'); ?>
                                    </label>
                                </div>
                            <?php else: ?>
                                <label class="pufsm-check">
                                    <input id="pufsm-<?php echo esc_attr($name); ?>" type="checkbox" name="pufsm[<?php echo esc_attr($name); ?>]" value="1">
                                    <span><?php echo esc_html($placeholder ? $placeholder : 'تأیید'); ?></span>
                                </label>
                            <?php endif; ?>
                        <?php else: ?>
                            <input id="pufsm-<?php echo esc_attr($name); ?>" type="<?php echo esc_attr($type); ?>" name="pufsm[<?php echo esc_attr($name); ?>]" <?php echo $required ? 'required' : ''; ?> placeholder="<?php echo esc_attr($placeholder); ?>" class="<?php echo esc_attr($input_class); ?>">
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>

                <div class="<?php echo esc_attr($field_wrap_class); ?>">
                    <label class="<?php echo esc_attr($label_class); ?>" for="pufsm-captcha">
                        کپچا <span class="pufsm-req">*</span>
                    </label>

                    <?php if ($mode === 'bootstrap'): ?>
                        <div class="<?php echo esc_attr($captcha_row_class); ?>">
                            <span class="<?php echo esc_attr($captcha_q_class); ?>"><span class="pufsm-cap-q"><?php echo esc_html($captcha['question']); ?></span> <span class="pufsm-cap-eq">=</span></span>
                            <input id="pufsm-captcha" type="text" name="pufsm_captcha" inputmode="numeric" autocomplete="off" required placeholder="پاسخ را وارد کنید" class="<?php echo esc_attr($input_class); ?>">
                            <input type="hidden" name="pufsm_cap_a" value="<?php echo esc_attr((int)$captcha['a']); ?>">
                            <input type="hidden" name="pufsm_cap_b" value="<?php echo esc_attr((int)$captcha['b']); ?>">
                            <input type="hidden" name="pufsm_cap_op" value="<?php echo esc_attr($captcha['op']); ?>">
                            <input type="hidden" name="pufsm_cap_ts" value="<?php echo esc_attr((int)$captcha['ts']); ?>">
                            <input type="hidden" name="pufsm_cap_token" value="<?php echo esc_attr($captcha['token']); ?>">
                        </div>
                    <?php else: ?>
                        <div class="<?php echo esc_attr($captcha_row_class); ?>">
                            <div class="<?php echo esc_attr($captcha_q_class); ?>" aria-label="سؤال کپچا"><span class="pufsm-cap-q"><?php echo esc_html($captcha['question']); ?></span> <span class="pufsm-cap-eq">=</span></div>
                            <input id="pufsm-captcha" type="text" name="pufsm_captcha" inputmode="numeric" autocomplete="off" required placeholder="پاسخ را وارد کنید" class="<?php echo esc_attr($input_class); ?>">
                            <input type="hidden" name="pufsm_cap_a" value="<?php echo esc_attr((int)$captcha['a']); ?>">
                            <input type="hidden" name="pufsm_cap_b" value="<?php echo esc_attr((int)$captcha['b']); ?>">
                            <input type="hidden" name="pufsm_cap_op" value="<?php echo esc_attr($captcha['op']); ?>">
                            <input type="hidden" name="pufsm_cap_ts" value="<?php echo esc_attr((int)$captcha['ts']); ?>">
                            <input type="hidden" name="pufsm_cap_token" value="<?php echo esc_attr($captcha['token']); ?>">
                        </div>
                    <?php endif; ?>
                </div>

                <div class="pufsm-actions">
                    <button type="submit" class="<?php echo esc_attr($btn_class); ?>">ارسال</button>
                </div>
            </form>
        </div>
        <?php
        return (string) ob_get_clean();
    }
}
