<?php
if (!defined('ABSPATH')) { exit; }

class PUFSM_Admin {
public static function init() {
        add_action('admin_menu', array(__CLASS__, 'menu'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'assets'));
        add_action('admin_init', array(__CLASS__, 'start_buffer'), 0);
        add_action('admin_post_pufsm_export_logs', array(__CLASS__, 'handle_export_logs'));
        add_action('admin_post_pufsm_delete_logs', array(__CLASS__, 'handle_delete_logs'));
}
public static function assets($hook) {
        if (strpos((string)$hook, 'pufsm') === false) { return; }

        wp_enqueue_style('pufsm-admin', PUFSM_PLUGIN_URL . 'assets/css/admin.css', array(), PUFSM_VERSION);

        // Builder (drag & drop) scripts
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_script(
            'pufsm-admin-builder',
            PUFSM_PLUGIN_URL . 'assets/js/admin-builder.js',
            array('jquery', 'jquery-ui-sortable'),
            PUFSM_VERSION,
            true
        );

        wp_localize_script('pufsm-admin-builder', 'PUFSM_ADMIN', array(
            'templates' => self::form_templates(),
            'i18n' => array(
                'confirm_delete_field' => 'این فیلد حذف شود؟',
                'field_label' => 'عنوان',
                'field_name' => 'نام (کلید)',
                'field_type' => 'نوع',
                'field_required' => 'اجباری',
                'field_placeholder' => 'Placeholder',
                'field_options' => 'گزینه‌ها (هر خط یک گزینه)',
            ),
        ));
    }
public static function menu() {
        add_menu_page(
            'فرم‌ساز پارس گیت',
            'پارس گیت',
            'manage_options',
            'pufsm',
            array(__CLASS__, 'page_forms'),
            'dashicons-email-alt2',
            58
        );

        add_submenu_page('pufsm', 'فرم‌ها', 'فرم‌ها', 'manage_options', 'pufsm', array(__CLASS__, 'page_forms'));
        add_submenu_page('pufsm', 'تنظیمات SMTP', 'تنظیمات SMTP', 'manage_options', 'pufsm-settings', array(__CLASS__, 'page_settings'));
        add_submenu_page('pufsm', 'گزارش‌ها', 'گزارش‌ها', 'manage_options', 'pufsm-reports', array(__CLASS__, 'page_reports'));
    }

    private static function admin_url($slug, $args=array()) {
        $url = admin_url('admin.php?page=' . $slug);
        if (!empty($args)) {
            $url = add_query_arg($args, $url);
        }
        return $url;
    }

    public static function start_buffer() {
        // Prevent "headers already sent" notices on our admin pages if any plugin/theme outputs early
        if (!is_admin()) { return; }
        $page = isset($_GET['page']) ? (string) $_GET['page'] : '';
        if ($page === '' || strpos($page, 'pufsm') === false) { return; }
        if (ob_get_level() === 0) {
            @ob_start();
        }
    }

    private static function safe_redirect($url) {
        // Always exit after redirect attempt
        if (!headers_sent()) {
            wp_safe_redirect($url);
            exit;
        }
        // Fallback: JS redirect (avoids warnings)
        echo '<script>window.location.href=' . wp_json_encode($url) . ';</script>';
        exit;
    }

    public static function footer_note() {
        return '<div class="pufsm-footer-note" dir="rtl">این افزونه طراحی و پیاده سازی شده برای مشترکین ایمیل سازمانی پارس آپتایم.</div>';
    }

    private static function build_logs_filters($status, $q, $form_id) {
        global $wpdb;
        $where = 'WHERE 1=1';
        $params = array();

        if ($status === 'sent') {
            $where .= ' AND l.status=1';
        } elseif ($status === 'failed') {
            $where .= ' AND l.status=0';
        }

        if (!empty($form_id)) {
            $where .= ' AND l.form_id=%d';
            $params[] = (int) $form_id;
        }

        if (!empty($q)) {
            $like = '%' . $wpdb->esc_like($q) . '%';
            $where .= ' AND (l.subject LIKE %s OR l.to_email LIKE %s OR l.sender_email LIKE %s OR l.payload LIKE %s)';
            array_push($params, $like, $like, $like, $like);
        }

        return array($where, $params);
    }

    public static function handle_export_logs() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden', 403); }
        check_admin_referer('pufsm_export_logs');

        $format = isset($_GET['format']) ? sanitize_key($_GET['format']) : 'csv';
        $status = isset($_GET['status']) ? sanitize_key($_GET['status']) : 'all';
        $q = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $form_id = isset($_GET['form_id']) ? (int) $_GET['form_id'] : 0;

        global $wpdb;
        $logs_table  = $wpdb->prefix . 'pufsm_logs';
        $forms_table = $wpdb->prefix . 'pufsm_forms';

        list($where, $params) = self::build_logs_filters($status, $q, $form_id);

        $sql = "SELECT l.*, f.name AS form_name
                FROM $logs_table l
                LEFT JOIN $forms_table f ON f.id = l.form_id
                $where
                ORDER BY l.id DESC
                LIMIT 50000";

        if (!empty($params)) {
            $sql = $wpdb->prepare($sql, $params);
        }

        $rows = $wpdb->get_results($sql, ARRAY_A);
        $now = gmdate('Ymd-His');

        if ($format === 'json') {
            nocache_headers();
            header('Content-Type: application/json; charset=utf-8');
            header('Content-Disposition: attachment; filename=parsgate-logs-' . $now . '.json');

            $out = array();
            foreach ($rows as $r) {
                $payload = array();
                if (!empty($r['payload'])) {
                    $payload = json_decode((string)$r['payload'], true);
                    if (!is_array($payload)) { $payload = array(); }
                }
                $out[] = array(
                    'id' => (int) $r['id'],
                    'created_at' => (string) $r['created_at'],
                    'status' => !empty($r['status']) ? 'sent' : 'failed',
                    'form' => (string) ($r['form_name'] ? $r['form_name'] : ''),
                    'to' => (string) $r['to_email'],
                    'sender' => (string) $r['sender_email'],
                    'subject' => (string) $r['subject'],
                    'fields' => isset($payload['fields']) ? $payload['fields'] : $payload,
                    'error' => (string) $r['error_text'],
                    'page_url' => (string) $r['page_url'],
                );
            }
            echo wp_json_encode($out, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
            exit;
        }

        // CSV (default)
        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=parsgate-logs-' . $now . '.csv');

        // UTF-8 BOM for Excel
        echo chr(239) . chr(187) . chr(191);

        $fh = fopen('php://output', 'w');
        fputcsv($fh, array('ID','Date','Status','Form','To','Sender','Subject','Message','Error','Page'));

        foreach ($rows as $r) {
            $payload = array();
            if (!empty($r['payload'])) {
                $payload = json_decode((string)$r['payload'], true);
                if (!is_array($payload)) { $payload = array(); }
            }

            $msg = '';
            $fields = isset($payload['fields']) && is_array($payload['fields']) ? $payload['fields'] : array();
            if (!empty($fields)) {
                foreach ($fields as $k => $v) {
                    $label = isset($v['label']) ? (string)$v['label'] : (string)$k;
                    $value = isset($v['value']) ? (string)$v['value'] : '';
                    $msg .= $label . ': ' . str_replace(array("\r","\n"), ' ', $value) . " | ";
                }
                $msg = rtrim($msg, " |");
            } elseif (!empty($payload['message'])) {
                $msg = (string) $payload['message'];
            }

            fputcsv($fh, array(
                (int) $r['id'],
                (string) $r['created_at'],
                !empty($r['status']) ? 'sent' : 'failed',
                (string) ($r['form_name'] ? $r['form_name'] : ''),
                (string) $r['to_email'],
                (string) $r['sender_email'],
                (string) $r['subject'],
                (string) $msg,
                (string) $r['error_text'],
                (string) $r['page_url'],
            ));
        }

        fclose($fh);
        exit;
    }

    public static function handle_delete_logs() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden', 403); }
        check_admin_referer('pufsm_delete_logs');

        $mode = isset($_GET['mode']) ? sanitize_key($_GET['mode']) : 'filtered';
        $status = isset($_GET['status']) ? sanitize_key($_GET['status']) : 'all';
        $q = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $form_id = isset($_GET['form_id']) ? (int) $_GET['form_id'] : 0;

        global $wpdb;
        $logs_table = $wpdb->prefix . 'pufsm_logs';

        $deleted = 0;

        if ($mode === 'all') {
            $deleted = (int) $wpdb->query("DELETE FROM $logs_table");
        } else {
            list($where, $params) = self::build_logs_filters($status, $q, $form_id);

            // Convert alias l. to table name for DELETE
            $where = str_replace('l.', '', $where);

            $sql = "DELETE FROM $logs_table $where";
            if (!empty($params)) {
                $sql = $wpdb->prepare($sql, $params);
            }
            $deleted = (int) $wpdb->query($sql);
        }

        $back = admin_url('admin.php?page=pufsm-reports');
        $back = add_query_arg(array(
            'status' => $status,
            'q' => $q,
            'form_id' => $form_id,
            'deleted' => max(0, $deleted),
        ), $back);

        self::safe_redirect($back);
    }


    private static function form_templates() {
        return array(
            'contact_pro' => array(
                array('name'=>'first_name','label'=>'نام','type'=>'text','required'=>true,'placeholder'=>'نام'),
                array('name'=>'last_name','label'=>'نام خانوادگی','type'=>'text','required'=>true,'placeholder'=>'نام خانوادگی'),
                array('name'=>'email','label'=>'ایمیل','type'=>'email','required'=>true,'placeholder'=>'name@example.com'),
                array('name'=>'mobile','label'=>'شماره موبایل','type'=>'tel','required'=>false,'placeholder'=>'09xxxxxxxxx'),
                array('name'=>'topic','label'=>'موضوع','type'=>'select','required'=>true,'options'=>array('مشاوره','پشتیبانی','سایر')),
                array('name'=>'message','label'=>'متن توضیحات','type'=>'textarea','required'=>true,'placeholder'=>'متن پیام...'),
            ),
            'contact_simple' => array(
                array('name'=>'full_name','label'=>'نام و نام خانوادگی','type'=>'text','required'=>true,'placeholder'=>'نام شما'),
                array('name'=>'email','label'=>'ایمیل','type'=>'email','required'=>true,'placeholder'=>'name@example.com'),
                array('name'=>'message','label'=>'پیام','type'=>'textarea','required'=>true,'placeholder'=>'متن پیام...'),
            ),
            'callback' => array(
                array('name'=>'first_name','label'=>'نام','type'=>'text','required'=>true,'placeholder'=>'نام'),
                array('name'=>'last_name','label'=>'نام خانوادگی','type'=>'text','required'=>true,'placeholder'=>'نام خانوادگی'),
                array('name'=>'mobile','label'=>'شماره موبایل','type'=>'tel','required'=>true,'placeholder'=>'09xxxxxxxxx'),
                array('name'=>'best_time','label'=>'زمان پیشنهادی تماس','type'=>'select','required'=>false,'options'=>array('صبح','ظهر','عصر','شب')),
                array('name'=>'notes','label'=>'توضیحات','type'=>'textarea','required'=>false,'placeholder'=>'...'),
            ),
            'support' => array(
                array('name'=>'full_name','label'=>'نام و نام خانوادگی','type'=>'text','required'=>true,'placeholder'=>'نام شما'),
                array('name'=>'email','label'=>'ایمیل','type'=>'email','required'=>true,'placeholder'=>'name@example.com'),
                array('name'=>'service','label'=>'سرویس','type'=>'select','required'=>false,'options'=>array('هاست','سرور مجازی','ایمیل سازمانی','دامنه','سایر')),
                array('name'=>'subject','label'=>'موضوع','type'=>'text','required'=>true,'placeholder'=>'موضوع درخواست'),
                array('name'=>'message','label'=>'شرح مشکل','type'=>'textarea','required'=>true,'placeholder'=>'...'),
            ),
        );
    }



    public static function page_settings() {
        if (!current_user_can('manage_options')) { return; }

        $settings = PUFSM_Settings::get();

        // Handle test email
        if (!empty($_POST['pufsm_test_send']) && check_admin_referer('pufsm_test_mail')) {
            $to = isset($_POST['pufsm_test_to']) ? sanitize_email(wp_unslash($_POST['pufsm_test_to'])) : '';
            if (!$to || !is_email($to)) {
                add_settings_error('pufsm', 'pufsm_test_to', 'ایمیل تست معتبر نیست.', 'error');
            } else {
                $ok = false;
                $err = '';
                try {
                    $ok = PUFSM_Mailer::send($to, 'Test Email - Parsgate Forms', '<div dir="rtl" style="font-family:Tahoma">این یک ایمیل تست از افزونه پارس گیت است.</div>', array('Content-Type: text/html; charset=UTF-8'));
                    if (!$ok) { $err = 'wp_mail returned false.'; }
                } catch (\Throwable $e) {
                    $ok = false;
                    $err = $e->getMessage();
                }
                if ($ok) {
                    add_settings_error('pufsm', 'pufsm_test_ok', 'ایمیل تست با موفقیت ارسال شد.', 'updated');
                } else {
                    add_settings_error('pufsm', 'pufsm_test_fail', 'ارسال ایمیل تست ناموفق بود: ' . esc_html($err), 'error');
                }
            }
        }

        settings_errors('pufsm');
        ?>
        <div class="wrap pufsm-admin" dir="rtl">
            <h1>تنظیمات SMTP</h1>

            <form method="post" action="options.php">
                <?php
                settings_fields('pufsm_settings_group');
                $opt = PUFSM_Settings::OPT_KEY;
                ?>

                <div class="pufsm-card">
                    <h2>اتصال SMTP</h2>

                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="smtp_host">Host</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[smtp_host]" id="smtp_host" type="text" class="regular-text" value="<?php echo esc_attr($settings['smtp_host']); ?>"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="smtp_port">Port</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[smtp_port]" id="smtp_port" type="number" class="small-text" value="<?php echo esc_attr((int)$settings['smtp_port']); ?>"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="smtp_secure">Encryption</label></th>
                            <td>
                                <select name="<?php echo esc_attr($opt); ?>[smtp_secure]" id="smtp_secure">
                                    <option value="tls" <?php selected($settings['smtp_secure'], 'tls'); ?>>TLS (STARTTLS)</option>
                                    <option value="ssl" <?php selected($settings['smtp_secure'], 'ssl'); ?>>SSL</option>
                                    <option value="none" <?php selected($settings['smtp_secure'], 'none'); ?>>None</option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">SMTP Auth</th>
                            <td>
                                <label><input type="checkbox" name="<?php echo esc_attr($opt); ?>[smtp_auth]" value="1" <?php checked(!empty($settings['smtp_auth'])); ?>> فعال</label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="smtp_user">شناسه اتصال</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[smtp_user]" id="smtp_user" type="text" class="regular-text" value="<?php echo esc_attr($settings['smtp_user']); ?>"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="smtp_pass">کلید اتصال</label></th>
                            <td>
                                <input name="<?php echo esc_attr($opt); ?>[smtp_pass]" id="smtp_pass" type="password" class="regular-text" value="" autocomplete="new-password" placeholder="برای عدم تغییر، خالی بگذارید">
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="pufsm-card">
                    <h2>فرستنده و پاسخ</h2>
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="from_email">ایمیل فرستنده</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[from_email]" id="from_email" type="email" class="regular-text" value="<?php echo esc_attr($settings['from_email']); ?>" placeholder="اگر خالی باشد، از شناسه اتصال استفاده می‌شود (مثلاً info@example.com)"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="from_name">نام فرستنده</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[from_name]" id="from_name" type="text" class="regular-text" value="<?php echo esc_attr($settings['from_name']); ?>"></td>
                        </tr>
                        <tr>
                            <th scope="row">Reply-To</th>
                            <td>
                                <label><input type="checkbox" name="<?php echo esc_attr($opt); ?>[reply_to_use_sender]" value="1" <?php checked(!empty($settings['reply_to_use_sender'])); ?>>
                                    اگر در فرم فیلد Email وجود داشت، Reply-To برابر ایمیل ارسال‌کننده شود
                                </label>
                            </td>
                        </tr>
                    </table>
                </div>

                
                <div class="pufsm-card">
                    <h2>استایل فرم</h2>
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="style_mode">حالت استایل</label></th>
                            <td>
                                <select name="<?php echo esc_attr($opt); ?>[style_mode]" id="style_mode">
                                    <option value="theme" <?php selected($settings['style_mode'], 'theme'); ?>>سازگار با قالب (پیشنهادی)</option>
                                    <option value="bootstrap" <?php selected($settings['style_mode'], 'bootstrap'); ?>>Bootstrap 5</option>
                                    <option value="none" <?php selected($settings['style_mode'], 'none'); ?>>خام (بدون استایل)</option>
                                </select>
                                <p class="description">اگر قالب شما Bootstrap دارد، حالت Bootstrap را انتخاب کنید. در حالت خام، فقط کلاس‌ها خروجی می‌شوند و CSS بارگذاری نمی‌شود.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="wrapper_class">کلاس Wrapper</label></th>
                            <td>
                                <input name="<?php echo esc_attr($opt); ?>[wrapper_class]" id="wrapper_class" type="text" class="regular-text" value="<?php echo esc_attr($settings['wrapper_class']); ?>" placeholder="مثال: my-form-wrapper">
                                <p class="description">به Wrapper اصلی فرم اضافه می‌شود.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="input_class">کلاس فیلدها</label></th>
                            <td>
                                <input name="<?php echo esc_attr($opt); ?>[input_class]" id="input_class" type="text" class="regular-text" value="<?php echo esc_attr($settings['input_class']); ?>" placeholder="مثال: form-control">
                                <p class="description">به input/textarea/select اضافه می‌شود. (برای Bootstrap معمولاً form-control و برای select: form-select)</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="button_class">کلاس دکمه</label></th>
                            <td>
                                <input name="<?php echo esc_attr($opt); ?>[button_class]" id="button_class" type="text" class="regular-text" value="<?php echo esc_attr($settings['button_class']); ?>" placeholder="مثال: btn btn-primary w-100">
                                <p class="description">به دکمه ارسال اضافه می‌شود.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="custom_css">CSS سفارشی</label></th>
                            <td>
                                <textarea name="<?php echo esc_attr($opt); ?>[custom_css]" id="custom_css" rows="10" class="large-text code" dir="ltr" style="unicode-bidi:embed;"><?php echo esc_textarea($settings['custom_css']); ?></textarea>
                                <p class="description">در فرانت‌اند فقط زمانی که فرم نمایش داده شود، لود می‌شود.</p>
                            </td>
                        </tr>
                    </table>
                </div>

<div class="pufsm-card">
                    <h2>محدودسازی ارسال (ضد اسپم)</h2>
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="rate_limit_seconds">حداقل فاصله بین ارسال‌ها (ثانیه)</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[rate_limit_seconds]" id="rate_limit_seconds" type="number" class="small-text" value="<?php echo esc_attr((int)$settings['rate_limit_seconds']); ?>"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="rate_limit_max_per_hour">حداکثر ارسال از هر IP در ۱ ساعت</label></th>
                            <td><input name="<?php echo esc_attr($opt); ?>[rate_limit_max_per_hour]" id="rate_limit_max_per_hour" type="number" class="small-text" value="<?php echo esc_attr((int)$settings['rate_limit_max_per_hour']); ?>"></td>
                        </tr>
                    </table>
                </div>

                <?php submit_button('ذخیره تنظیمات'); ?>
            </form>

            <div class="pufsm-card">
                <h2>ارسال ایمیل تست</h2>
                <form method="post">
                    <?php wp_nonce_field('pufsm_test_mail'); ?>
                    <input type="hidden" name="pufsm_test_send" value="1">
                    <p>
                        <label for="pufsm_test_to">ارسال به:</label>
                        <input id="pufsm_test_to" name="pufsm_test_to" type="email" class="regular-text" value="<?php echo esc_attr(get_option('admin_email')); ?>">
                        <button class="button button-primary">ارسال تست</button>
                    </p>
                </form>
                <p class="description">قبل از تست، تنظیمات را ذخیره کنید.</p>
            </div>
        </div>
            <?php echo self::footer_note(); ?>

        <?php
    }
public static function page_forms() {
        if (!current_user_can('manage_options')) { return; }

        $action  = isset($_GET['action']) ? sanitize_key($_GET['action']) : '';
        $form_id = isset($_GET['form_id']) ? (int) $_GET['form_id'] : 0;

        // Delete
        if ($action === 'delete' && $form_id && check_admin_referer('pufsm_delete_form_' . $form_id)) {
            PUFSM_Forms::delete_form($form_id);
            self::safe_redirect(self::admin_url('pufsm', array('deleted'=>1)));
        }

        $notice = '';
        $error  = '';

        if (isset($_GET['deleted'])) {
            $notice = 'تعداد ' . (int) $_GET['deleted'] . ' لاگ حذف شد.';
        }
// Save (create/update)
        if (!empty($_POST['pufsm_save_form']) && check_admin_referer('pufsm_save_form')) {
            $data = self::sanitize_form_post();
            if ($data['ok']) {
                if (!empty($_POST['form_id'])) {
                    PUFSM_Forms::update_form((int)$_POST['form_id'], $data['data']);
                    $form_id = (int)$_POST['form_id'];
                    $action = 'edit';
                } else {
                    $new_id = PUFSM_Forms::create_form($data['data']);
                    if ($new_id) {
                        $form_id = (int)$new_id;
                        $action = 'edit';
                    }
                }
                $notice = 'تغییرات ذخیره شد.';
            } else {
                $error = $data['error'];
            }
        }

        if (isset($_GET['deleted'])) { $notice = 'فرم حذف شد.'; }

        // Edit / Add form
        if ($action === 'add' || ($action === 'edit' && $form_id)) {
            $form = ($action === 'edit') ? PUFSM_Forms::get_form($form_id) : null;
            if ($action === 'edit' && !$form) {
                echo '<div class="wrap"><div class="notice notice-error"><p>فرم پیدا نشد.</p></div></div>';
                return;
            }

            $name = $form ? $form['name'] : '';
            $recipient = $form ? $form['recipient'] : '';
            $subject = $form ? $form['subject'] : 'پیام جدید از فرم: {form_name}';
            $success_message = $form ? $form['success_message'] : 'پیام شما با موفقیت ارسال شد.';
            $enabled = $form ? (int)$form['is_enabled'] : 1;

            $fields_json = $form ? (string)$form['fields'] : wp_json_encode(array(
                array("name"=>"first_name","label"=>"نام","type"=>"text","required"=>true, "placeholder"=>"نام"),
                array("name"=>"last_name","label"=>"نام خانوادگی","type"=>"text","required"=>true, "placeholder"=>"نام خانوادگی"),
                array("name"=>"mobile","label"=>"شماره موبایل","type"=>"tel","required"=>false, "placeholder"=>"09xxxxxxxxx"),
                array("name"=>"topic","label"=>"موضوع","type"=>"select","required"=>true, "options"=>array("مشاوره","پشتیبانی","سایر")),
                array("name"=>"message","label"=>"متن توضیحات","type"=>"textarea","required"=>true, "placeholder"=>"متن پیام..."),
            ), JSON_UNESCAPED_UNICODE);

            ?>
            <div class="wrap pufsm-admin" dir="rtl">
                <h1><?php echo ($action === 'edit') ? 'ویرایش فرم' : 'افزودن فرم'; ?></h1>

                <?php if ($notice): ?>
                    <div class="notice notice-success is-dismissible"><p><?php echo esc_html($notice); ?></p></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="notice notice-error is-dismissible"><p><?php echo esc_html($error); ?></p></div>
                <?php endif; ?>

                <form method="post" id="pufsm-form-editor">
                    <?php wp_nonce_field('pufsm_save_form'); ?>
                    <input type="hidden" name="pufsm_save_form" value="1">
                    <?php if ($action === 'edit'): ?>
                        <input type="hidden" name="form_id" value="<?php echo esc_attr((int)$form_id); ?>">
                    <?php endif; ?>

                    <div class="pufsm-card">
                        <table class="form-table" role="presentation">
                            <tr>
                                <th scope="row">نام فرم</th>
                                <td><input name="name" type="text" class="regular-text" value="<?php echo esc_attr($name); ?>" required></td>
                            </tr>
                            <tr>
                                <th scope="row">ارسال به (Recipient)</th>
                                <td><input name="recipient" type="email" class="regular-text" value="<?php echo esc_attr($recipient); ?>" required></td>
                            </tr>
                            <tr>
                                <th scope="row">Subject ایمیل</th>
                                <td><input name="subject" type="text" class="regular-text" value="<?php echo esc_attr($subject); ?>" required>
                                    <p class="description">می‌توانید از <code dir="ltr">{form_name}</code> و <code dir="ltr">{page_url}</code> استفاده کنید.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">پیام موفقیت</th>
                                <td><input name="success_message" type="text" class="regular-text" value="<?php echo esc_attr($success_message); ?>" required></td>
                            </tr>
                            <tr>
                                <th scope="row">فعال</th>
                                <td><label><input name="is_enabled" type="checkbox" value="1" <?php checked($enabled, 1); ?>> فعال باشد</label></td>
                            </tr>
                        </table>
                    </div>

                    <?php if ($action === 'edit' && $form_id): ?>
                        <div class="pufsm-card pufsm-shortcode-card">
                            <div class="pufsm-row">
                                <strong>شورت‌کد این فرم:</strong>
                                <input type="text" class="regular-text pufsm-copy-input" readonly value="[pu_form id=&quot;<?php echo esc_attr((int)$form_id); ?>&quot;]">
                                <button type="button" class="button pufsm-copy-btn" data-copy="[pu_form id=&quot;<?php echo esc_attr((int)$form_id); ?>&quot;]">کپی</button>
                            </div>
                            <p class="description">این شورت‌کد را در هر برگه/صفحه/المنتور قرار دهید.</p>
                        </div>
                    <?php endif; ?>

                    <div class="pufsm-card">
                        <h2 style="margin:0 0 12px 0;">قالب‌های آماده</h2>
                        <div class="pufsm-row" style="gap:10px; align-items:center;">
                            <select id="pufsm-template" class="regular-text">
                                <option value="">انتخاب کنید...</option>
                                <option value="contact_pro">فرم تماس کامل</option>
                                <option value="contact_simple">فرم تماس ساده</option>
                                <option value="callback">فرم درخواست تماس</option>
                                <option value="support">فرم پشتیبانی</option>
                            </select>
                            <button type="button" class="button" id="pufsm-apply-template">اعمال قالب</button>
                            <span class="description">قالب را انتخاب کنید تا فیلدها به‌صورت خودکار ساخته شوند.</span>
                        </div>

                        <hr>

                        <h2 style="margin:0 0 12px 0;">سازنده فیلدها</h2>

                        <input type="hidden" id="pufsm-fields-initial" value="<?php echo esc_attr($fields_json); ?>">

                        <table class="widefat striped pufsm-builder-table" id="pufsm-fields-table">
                            <thead>
                                <tr>
                                    <th style="width:28px;"></th>
                                    <th>عنوان</th>
                                    <th>نام (کلید)</th>
                                    <th style="width:130px;">نوع</th>
                                    <th style="width:70px;">اجباری</th>
                                    <th>Placeholder</th>
                                    <th>گزینه‌ها</th>
                                    <th style="width:80px;">عملیات</th>
                                </tr>
                            </thead>
                            <tbody></tbody>
                        </table>

                        <p style="margin-top:12px;">
                            <button type="button" class="button" id="pufsm-add-field">افزودن فیلد</button>
                        </p>

                        <details style="margin-top:12px;">
                            <summary>نمایش/ویرایش JSON (پیشرفته)</summary>
                            <textarea name="fields" class="large-text code" rows="10" id="pufsm-fields-json" required><?php echo esc_textarea($fields_json); ?></textarea>
                            <p class="description">
                                typeهای مجاز: text, email, textarea, tel, number, select, checkbox
                            </p>
                        </details>
                    </div>

                    <p class="submit">
                        <button type="submit" class="button button-primary">ذخیره</button>
                        <a class="button" href="<?php echo esc_url(self::admin_url('pufsm')); ?>">بازگشت</a>
                    </p>
                </form>
            </div>
            <?php
            return;
        }

        // List forms
        $forms = PUFSM_Forms::list_forms();
        ?>
        <div class="wrap pufsm-admin" dir="rtl">
            <h1>فرم‌ها</h1>

            <?php if ($notice): ?>
                <div class="notice notice-success is-dismissible"><p><?php echo esc_html($notice); ?></p></div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="notice notice-error is-dismissible"><p><?php echo esc_html($error); ?></p></div>
            <?php endif; ?>

            <p>
                <a class="button button-primary" href="<?php echo esc_url(self::admin_url('pufsm', array('action'=>'add'))); ?>">افزودن فرم جدید</a>
            </p>

            <div class="pufsm-card">
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th style="width:70px;">ID</th>
                            <th>نام فرم</th>
                            <th>Recipient</th>
                            <th style="width:80px;">وضعیت</th>
                            <th>شورت‌کد</th>
                            <th style="width:160px;">عملیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($forms)): foreach ($forms as $f): ?>
                            <tr>
                                <td><?php echo esc_html((int)$f['id']); ?></td>
                                <td><?php echo esc_html($f['name']); ?></td>
                                <td><?php echo esc_html($f['recipient']); ?></td>
                                <td><?php echo !empty($f['is_enabled']) ? '<span class="pufsm-badge pufsm-on">فعال</span>' : '<span class="pufsm-badge pufsm-off">غیرفعال</span>'; ?></td>
                                <td>
                                    <code dir="ltr">[pu_form id="<?php echo esc_attr((int)$f['id']); ?>"]</code>
                                    <button type="button" class="button pufsm-copy-btn" data-copy="[pu_form id=&quot;<?php echo esc_attr((int)$f['id']); ?>&quot;]">کپی</button>
                                </td>
                                <td>
                                    <a class="button" href="<?php echo esc_url(self::admin_url('pufsm', array('action'=>'edit','form_id'=>(int)$f['id']))); ?>">ویرایش</a>
                                    <a class="button button-link-delete" href="<?php echo esc_url(wp_nonce_url(self::admin_url('pufsm', array('action'=>'delete','form_id'=>(int)$f['id'])), 'pufsm_delete_form_' . (int)$f['id'])); ?>" onclick="return confirm('این فرم حذف شود؟');">حذف</a>
                                </td>
                            </tr>
                        <?php endforeach; else: ?>
                            <tr><td colspan="6">هنوز فرمی ساخته نشده است.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
            <?php echo self::footer_note(); ?>

        <?php
    }

    private static function sanitize_form_post() {
        $name = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
        $recipient = isset($_POST['recipient']) ? sanitize_email(wp_unslash($_POST['recipient'])) : '';
        $subject = isset($_POST['subject']) ? sanitize_text_field(wp_unslash($_POST['subject'])) : '';
        $success_message = isset($_POST['success_message']) ? sanitize_text_field(wp_unslash($_POST['success_message'])) : '';
        $is_enabled = !empty($_POST['is_enabled']) ? 1 : 0;
        $fields_raw = isset($_POST['fields']) ? wp_unslash($_POST['fields']) : '';

        if ($name === '' || $recipient === '' || $subject === '' || $success_message === '') {
            return array('ok'=>false,'error'=>'همه فیلدهای ضروری را تکمیل کنید.');
        }
        if (!is_email($recipient)) {
            return array('ok'=>false,'error'=>'Recipient معتبر نیست.');
        }

        $decoded = json_decode($fields_raw, true);
        if (!is_array($decoded)) {
            return array('ok'=>false,'error'=>'JSON فیلدها معتبر نیست.');
        }

        $allowed_types = array('text','email','textarea','tel','number','select','checkbox');
        $clean = array();
        foreach ($decoded as $f) {
            if (!is_array($f)) { continue; }
            $fname = isset($f['name']) ? sanitize_key($f['name']) : '';
            if (!$fname) { continue; }
            $ftype = isset($f['type']) ? sanitize_key($f['type']) : 'text';
            if (!in_array($ftype, $allowed_types, true)) { $ftype = 'text'; }

            $row = array(
                'name' => $fname,
                'label' => isset($f['label']) ? sanitize_text_field($f['label']) : $fname,
                'type' => $ftype,
                'required' => !empty($f['required']) ? true : false,
            );

            if (!empty($f['placeholder'])) {
                $row['placeholder'] = sanitize_text_field($f['placeholder']);
            }

            if ($ftype === 'select' && !empty($f['options']) && is_array($f['options'])) {
                $opts = array();
                foreach ($f['options'] as $opt) {
                    $opt = is_string($opt) ? sanitize_text_field($opt) : '';
                    if ($opt !== '') { $opts[] = $opt; }
                }
                $row['options'] = $opts;
            }

            $clean[] = $row;
        }

        if (empty($clean)) {
            return array('ok'=>false,'error'=>'حداقل یک فیلد معتبر لازم است.');
        }

        $fields_json = wp_json_encode($clean, JSON_UNESCAPED_UNICODE);
        return array('ok'=>true,'data'=>array(
            'name' => $name,
            'recipient' => $recipient,
            'subject' => $subject,
            'fields' => $fields_json,
            'success_message' => $success_message,
            'is_enabled' => $is_enabled,
        ));
    }
public static function page_reports() {
        if (!current_user_can('manage_options')) { return; }

        global $wpdb;
        $logs_table  = $wpdb->prefix . 'pufsm_logs';
        $forms_table = $wpdb->prefix . 'pufsm_forms';

        $notice = '';
        $error  = '';

        // View single message
        $view_id = isset($_GET['view']) ? (int) $_GET['view'] : 0;
        if ($view_id) {

            // Handle admin reply
            if (!empty($_POST['pufsm_send_reply']) && check_admin_referer('pufsm_send_reply_' . $view_id)) {
                $to      = isset($_POST['reply_to']) ? sanitize_email(wp_unslash($_POST['reply_to'])) : '';
                $subject = isset($_POST['reply_subject']) ? sanitize_text_field(wp_unslash($_POST['reply_subject'])) : '';
                $msg     = isset($_POST['reply_message']) ? wp_kses_post(wp_unslash($_POST['reply_message'])) : '';

                if (!$to || !is_email($to)) {
                    $error = 'ایمیل گیرنده (پاسخ) معتبر نیست.';
                } elseif ($subject === '' || wp_strip_all_tags($msg) === '') {
                    $error = 'موضوع و متن پاسخ نمی‌تواند خالی باشد.';
                } else {
                    $sent = false;
                    $errt = '';
                    try {
                        $sent = PUFSM_Mailer::send($to, $subject, wpautop($msg), array());
                        if (!$sent) { $errt = 'send returned false.'; }
                    } catch (\Throwable $e) {
                        $sent = false;
                        $errt = $e->getMessage();
                    }

                    // Log reply as a separate entry (payload marks it as admin reply)
                    $payload = array('admin_reply_for' => $view_id, 'message' => wp_strip_all_tags($msg));
                    PUFSM_Forms::log_send(0, $to, $subject, $sent ? 1 : 0, $errt, '', $payload);

                    if ($sent) {
                        $notice = 'پاسخ با موفقیت ارسال شد.';
                    } else {
                        $error = 'ارسال پاسخ ناموفق بود. ' . $errt;
                    }
                }
            }

            $row = $wpdb->get_row($wpdb->prepare(
                "SELECT l.*, f.name AS form_name
                 FROM $logs_table l
                 LEFT JOIN $forms_table f ON f.id = l.form_id
                 WHERE l.id=%d",
                $view_id
            ), ARRAY_A);

            if (!$row) {
                echo '<div class="wrap"><div class="notice notice-error"><p>پیام پیدا نشد.</p></div></div>';
                return;
            }

            // Parse payload fields
            $payload = array();
            if (!empty($row['payload'])) {
                $payload = json_decode((string)$row['payload'], true);
                if (!is_array($payload)) { $payload = array(); }
            }
            $fields = isset($payload['fields']) && is_array($payload['fields']) ? $payload['fields'] : array();

            ?>
            <div class="wrap pufsm-admin" dir="rtl">
                <h1>جزئیات پیام</h1>

                <?php if ($notice): ?>
                    <div class="notice notice-success is-dismissible"><p><?php echo esc_html($notice); ?></p></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="notice notice-error is-dismissible"><p><?php echo esc_html($error); ?></p></div>
                <?php endif; ?>

                <p><a class="button" href="<?php echo esc_url(self::admin_url('pufsm-reports')); ?>">بازگشت به گزارش‌ها</a></p>

                <div class="pufsm-card">
                    <table class="widefat striped">
                        <tbody>
                            <tr><th style="width:180px;">ID</th><td><?php echo esc_html((int)$row['id']); ?></td></tr>
                            <tr><th>تاریخ</th><td><?php echo esc_html($row['created_at']); ?></td></tr>
                            <tr><th>فرم</th><td><?php echo esc_html(!empty($row['form_name']) ? $row['form_name'] : ('Form #' . (int)$row['form_id'])); ?></td></tr>
                            <tr><th>وضعیت ارسال</th><td><?php echo !empty($row['status']) ? '<span class="pufsm-badge pufsm-on">موفق</span>' : '<span class="pufsm-badge pufsm-off">ناموفق</span>'; ?></td></tr>
                            <tr><th>گیرنده</th><td><?php echo esc_html($row['to_email']); ?></td></tr>
                            <tr><th>فرستنده (Email)</th><td><?php echo esc_html($row['sender_email']); ?></td></tr>
                            <tr><th>موضوع</th><td><?php echo esc_html($row['subject']); ?></td></tr>
                            <tr><th>صفحه</th><td><?php echo esc_html($row['page_url']); ?></td></tr>
                            <?php if (!empty($row['error_text'])): ?>
                                <tr><th>خطا</th><td><code dir="ltr"><?php echo esc_html($row['error_text']); ?></code></td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

                <?php if (!empty($fields)): ?>
                    <div class="pufsm-card">
                        <h2 style="margin:0 0 12px 0;">محتوای فرم</h2>
                        <table class="widefat striped">
                            <thead><tr><th style="width:220px;">فیلد</th><th>مقدار</th></tr></thead>
                            <tbody>
                            <?php foreach ($fields as $k => $v):
                                // Normalize field payload to array(label,value,type)
                                $entry = $v;

                                // If value is a JSON string, try to decode
                                if (is_string($entry)) {
                                    $decoded = json_decode($entry, true);
                                    if (is_array($decoded)) { $entry = $decoded; }
                                }

                                $label = is_array($entry) && isset($entry['label']) ? (string)$entry['label'] : (string)$k;
                                $type  = is_array($entry) && isset($entry['type']) ? (string)$entry['type'] : '';
                                $value = is_array($entry) && array_key_exists('value', $entry) ? $entry['value'] : $entry;

                                if (is_array($value) || is_object($value)) {
                                    $value = wp_json_encode($value, JSON_UNESCAPED_UNICODE);
                                } else {
                                    $value = (string)$value;
                                }

                                $value_trim = trim($value);
                                $is_email = ($type === 'email') || is_email($value_trim);
                            ?>
                                <tr>
                                    <td>
                                        <div style="font-weight:600;"><?php echo esc_html($label); ?></div>
                                        <div class="description" style="margin-top:2px;opacity:.8;"><?php echo esc_html($k); ?></div>
                                    </td>
                                    <td>
                                        <?php if ($value_trim === ''): ?>
                                            <span class="description">—</span>
                                        <?php elseif ($type === 'textarea'): ?>
                                            <?php echo nl2br(esc_html($value)); ?>
                                        <?php elseif ($is_email): ?>
                                            <span dir="ltr" style="unicode-bidi:embed;display:inline-block;"><?php echo esc_html($value_trim); ?></span>
                                        <?php else: ?>
                                            <?php echo esc_html($value); ?>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <div class="pufsm-card">
                    <h2 style="margin:0 0 12px 0;">ارسال پاسخ</h2>
                    <form method="post">
                        <?php wp_nonce_field('pufsm_send_reply_' . $view_id); ?>
                        <input type="hidden" name="pufsm_send_reply" value="1">
                        <table class="form-table" role="presentation">
                            <tr>
                                <th scope="row">به</th>
                                <td><input type="email" name="reply_to" class="regular-text" value="<?php echo esc_attr($row['sender_email']); ?>" required></td>
                            </tr>
                            <tr>
                                <th scope="row">موضوع</th>
                                <td><input type="text" name="reply_subject" class="regular-text" value="<?php echo esc_attr('Re: ' . $row['subject']); ?>" required></td>
                            </tr>
                            <tr>
                                <th scope="row">متن</th>
                                <td><textarea name="reply_message" class="large-text" rows="6" required></textarea></td>
                            </tr>
                        </table>
                        <p><button type="submit" class="button button-primary">ارسال پاسخ</button></p>
                    </form>
                </div>
            </div>
            <?php
            return;
        }

        // Filters
        $status = isset($_GET['status']) ? sanitize_key($_GET['status']) : 'all';
        if (!in_array($status, array('all','sent','fail'), true)) { $status = 'all'; }

        $q = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $filter_form_id = isset($_GET['form_id']) ? (int) $_GET['form_id'] : 0;

        $conds = array();
        $params = array();

        if ($status === 'sent') { $conds[] = "l.status=1"; }
        if ($status === 'fail') { $conds[] = "l.status=0"; }

        if ($filter_form_id) {
            $conds[] = "l.form_id=%d";
            $params[] = $filter_form_id;
        }

        if ($q !== '') {
            $like = '%' . $wpdb->esc_like($q) . '%';
            $conds[] = "(l.subject LIKE %s OR l.to_email LIKE %s OR l.sender_email LIKE %s OR l.payload LIKE %s OR f.name LIKE %s)";
            $params[] = $like; $params[] = $like; $params[] = $like; $params[] = $like; $params[] = $like;
        }

        $where = $conds ? ("WHERE " . implode(" AND ", $conds)) : "";

        $per_page = 30;
        $paged = isset($_GET['paged']) ? max(1, (int)$_GET['paged']) : 1;
        $offset = ($paged - 1) * $per_page;

        $total_sql = "SELECT COUNT(*) FROM $logs_table l LEFT JOIN $forms_table f ON f.id = l.form_id $where";
        $total = $params ? (int)$wpdb->get_var($wpdb->prepare($total_sql, $params)) : (int)$wpdb->get_var($total_sql);

        $list_sql = "SELECT l.*, f.name AS form_name
             FROM $logs_table l
             LEFT JOIN $forms_table f ON f.id = l.form_id
             $where
             ORDER BY l.id DESC
             LIMIT %d OFFSET %d";

        $list_params = $params ? $params : array();
        $list_params[] = $per_page;
        $list_params[] = $offset;

        $rows = $wpdb->get_results($wpdb->prepare($list_sql, $list_params), ARRAY_A);

        $sent_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM $logs_table WHERE status=1");
        $fail_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM $logs_table WHERE status=0");

        $forms = PUFSM_Forms::list_forms();

        ?>
        <div class="wrap pufsm-admin" dir="rtl">
            <h1>گزارش ارسال‌ها و پیام‌ها</h1>

            <div class="pufsm-stats">
                <div class="pufsm-stat"><strong><?php echo esc_html($sent_count); ?></strong><span>ارسال موفق</span></div>
                <div class="pufsm-stat"><strong><?php echo esc_html($fail_count); ?></strong><span>ارسال ناموفق</span></div>
                <div class="pufsm-stat"><strong><?php echo esc_html($total); ?></strong><span>نتایج فیلتر فعلی</span></div>
            </div>

            <div class="pufsm-card">

                <?php
                    $base_export = admin_url('admin-post.php?action=pufsm_export_logs');
                    $base_delete = admin_url('admin-post.php?action=pufsm_delete_logs');

                    $common_args = array(
                        'status' => $status,
                        'q' => $q,
                        'form_id' => $filter_form_id,
                    );

                    $export_csv = wp_nonce_url(add_query_arg(array_merge($common_args, array('format'=>'csv')), $base_export), 'pufsm_export_logs');
                    $export_json = wp_nonce_url(add_query_arg(array_merge($common_args, array('format'=>'json')), $base_export), 'pufsm_export_logs');

                    $delete_filtered = wp_nonce_url(add_query_arg(array_merge($common_args, array('mode'=>'filtered')), $base_delete), 'pufsm_delete_logs');
                    $delete_all = wp_nonce_url(add_query_arg(array('mode'=>'all'), $base_delete), 'pufsm_delete_logs');
                ?>
                <div class="pufsm-tools">
                    <a class="button" href="<?php echo esc_url($export_csv); ?>">دانلود CSV</a>
                    <a class="button" href="<?php echo esc_url($export_json); ?>">دانلود JSON</a>
                    <a class="button" href="<?php echo esc_url($delete_filtered); ?>" onclick="return confirm('لاگ‌های همین فیلتر حذف شود؟');">حذف لاگ‌های فیلتر</a>
                    <a class="button button-danger" href="<?php echo esc_url($delete_all); ?>" onclick="return confirm('تمام لاگ‌ها حذف شود؟ این کار قابل برگشت نیست.');">حذف همه لاگ‌ها</a>
                </div>
                <div class="pufsm-filters" style="display:flex; gap:8px; align-items:center; justify-content:space-between; flex-wrap:wrap;">
                    <div style="display:flex; gap:8px; flex-wrap:wrap;">
                        <a class="button <?php echo ($status==='all'?'button-primary':''); ?>" href="<?php echo esc_url(self::admin_url('pufsm-reports', array('status'=>'all','q'=>$q,'form_id'=>$filter_form_id))); ?>">همه</a>
                        <a class="button <?php echo ($status==='sent'?'button-primary':''); ?>" href="<?php echo esc_url(self::admin_url('pufsm-reports', array('status'=>'sent','q'=>$q,'form_id'=>$filter_form_id))); ?>">موفق</a>
                        <a class="button <?php echo ($status==='fail'?'button-primary':''); ?>" href="<?php echo esc_url(self::admin_url('pufsm-reports', array('status'=>'fail','q'=>$q,'form_id'=>$filter_form_id))); ?>">ناموفق</a>
                    </div>

                    <form method="get" style="display:flex; gap:8px; flex-wrap:wrap; align-items:center;">
                        <input type="hidden" name="page" value="pufsm-reports">
                        <input type="hidden" name="status" value="<?php echo esc_attr($status); ?>">
                        <input type="text" name="q" class="regular-text" placeholder="جستجو در Subject/ایمیل/محتوا..." value="<?php echo esc_attr($q); ?>">
                        <select name="form_id">
                            <option value="0">همه فرم‌ها</option>
                            <?php if (!empty($forms)): foreach ($forms as $f): ?>
                                <option value="<?php echo esc_attr((int)$f['id']); ?>" <?php selected($filter_form_id, (int)$f['id']); ?>>
                                    <?php echo esc_html($f['name']); ?>
                                </option>
                            <?php endforeach; endif; ?>
                        </select>
                        <button type="submit" class="button">اعمال</button>
                    </form>
                </div>

                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th style="width:70px;">ID</th>
                            <th style="width:170px;">تاریخ</th>
                            <th>فرم</th>
                            <th>گیرنده</th>
                            <th>فرستنده</th>
                            <th>موضوع</th>
                            <th style="width:90px;">وضعیت</th>
                            <th style="width:110px;">جزئیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($rows)): foreach ($rows as $r): ?>
                            <tr>
                                <td><?php echo esc_html((int)$r['id']); ?></td>
                                <td><?php echo esc_html($r['created_at']); ?></td>
                                <td><?php echo esc_html(!empty($r['form_name']) ? $r['form_name'] : ('Form #' . (int)$r['form_id'])); ?></td>
                                <td><?php echo esc_html($r['to_email']); ?></td>
                                <td><?php echo esc_html($r['sender_email']); ?></td>
                                <td><?php echo esc_html($r['subject']); ?></td>
                                <td><?php echo !empty($r['status']) ? '<span class="pufsm-badge pufsm-on">موفق</span>' : '<span class="pufsm-badge pufsm-off">ناموفق</span>'; ?></td>
                                <td><a class="button" href="<?php echo esc_url(self::admin_url('pufsm-reports', array('view'=>(int)$r['id']))); ?>">مشاهده</a></td>
                            </tr>
                        <?php endforeach; else: ?>
                            <tr><td colspan="8">هیچ موردی یافت نشد.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>

                <?php
                $total_pages = (int) ceil($total / $per_page);
                if ($total_pages > 1):
                    $base_args = array('status'=>$status,'q'=>$q,'form_id'=>$filter_form_id);
                    ?>
                    <div class="tablenav" style="margin-top:12px;">
                        <div class="tablenav-pages">
                            <?php
                            echo paginate_links(array(
                                'base' => add_query_arg(array_merge($base_args, array('paged'=>'%#%')), self::admin_url('pufsm-reports')),
                                'format' => '',
                                'prev_text' => '«',
                                'next_text' => '»',
                                'total' => $total_pages,
                                'current' => $paged,
                            ));
                            ?>
                        </div>
                    </div>
                <?php endif; ?>

            </div>
            <p class="description">این صفحه علاوه بر وضعیت ارسال، محتوای فرم (پیام‌ها) را هم نمایش می‌دهد.</p>
        </div>
            <?php echo self::footer_note(); ?>

        <?php
    }
}
