(function($){
  function safeJsonParse(str){
    try { return JSON.parse(str); } catch(e){ return null; }
  }

  function normalizeField(f){
    if (!f || typeof f !== 'object') return null;
    var out = {
      name: (f.name || '').toString().trim(),
      label: (f.label || '').toString().trim(),
      type: (f.type || 'text').toString().trim(),
      required: !!f.required,
      placeholder: (f.placeholder || '').toString(),
      options: Array.isArray(f.options) ? f.options.map(function(x){ return (x||'').toString(); }) : []
    };
    if (!out.type) out.type = 'text';
    return out;
  }

  function collectFields($tbody){
    var fields = [];
    $tbody.find('tr').each(function(){
      var $tr = $(this);
      var type = $tr.find('.pufsm-type').val() || 'text';
      var f = {
        label: $tr.find('.pufsm-label').val() || '',
        name: $tr.find('.pufsm-name').val() || '',
        type: type,
        required: $tr.find('.pufsm-required').prop('checked') ? true : false
      };

      var ph = $tr.find('.pufsm-placeholder').val() || '';
      if (ph) f.placeholder = ph;

      if (type === 'select') {
        var optText = $tr.find('.pufsm-options').val() || '';
        var opts = optText.split(/\r?\n/).map(function(x){ return x.trim(); }).filter(function(x){ return x.length; });
        if (opts.length) f.options = opts;
      }

      // Basic validation: need name
      f.name = (f.name || '').toString().trim();
      f.label = (f.label || '').toString().trim();
      if (f.name) fields.push(f);
    });
    return fields;
  }

  function syncJson($tbody, $json){
    var fields = collectFields($tbody);
    $json.val(JSON.stringify(fields, null, 2));
  }

  function renderRow(field){
    field = normalizeField(field) || normalizeField({type:'text'}) || {type:'text'};
    var optionsText = (field.options || []).join("\n");
    var tr = $(
      '<tr>' +
        '<td><span class="pufsm-drag" title="جابجایی">≡</span></td>' +
        '<td><input type="text" class="pufsm-label" value=""></td>' +
        '<td><input type="text" class="pufsm-name" value="" dir="ltr"></td>' +
        '<td>' +
          '<select class="pufsm-type">' +
            '<option value="text">text</option>' +
            '<option value="email">email</option>' +
            '<option value="tel">tel</option>' +
            '<option value="number">number</option>' +
            '<option value="textarea">textarea</option>' +
            '<option value="select">select</option>' +
            '<option value="checkbox">checkbox</option>' +
          '</select>' +
        '</td>' +
        '<td style="text-align:center;"><input type="checkbox" class="pufsm-required"></td>' +
        '<td><input type="text" class="pufsm-placeholder" value=""></td>' +
        '<td><textarea class="pufsm-options" placeholder="گزینه‌ها..."></textarea></td>' +
        '<td><button type="button" class="button pufsm-remove">حذف</button></td>' +
      '</tr>'
    );

    tr.find('.pufsm-label').val(field.label || '');
    tr.find('.pufsm-name').val(field.name || '');
    tr.find('.pufsm-type').val(field.type || 'text');
    tr.find('.pufsm-required').prop('checked', !!field.required);
    tr.find('.pufsm-placeholder').val(field.placeholder || '');
    tr.find('.pufsm-options').val(optionsText);

    return tr;
  }

  function toggleOptions($tr){
    var type = $tr.find('.pufsm-type').val() || 'text';
    var $optCell = $tr.find('.pufsm-options').closest('td');
    if (type === 'select') {
      $optCell.show();
    } else {
      $optCell.hide();
    }
  }

  function renderTable($tbody, fields){
    $tbody.empty();
    (fields || []).forEach(function(f){
      var $tr = renderRow(f);
      $tbody.append($tr);
      toggleOptions($tr);
    });
  }

  function initBuilder(){
    var $initial = $('#pufsm-fields-initial');
    var $json = $('#pufsm-fields-json');
    var $table = $('#pufsm-fields-table');
    if (!$table.length || !$json.length) return;

    var $tbody = $table.find('tbody');
    var fields = [];

    var initialStr = $initial.length ? $initial.val() : '';
    var parsed = initialStr ? safeJsonParse(initialStr) : null;
    if (!parsed) parsed = safeJsonParse($json.val() || '[]');
    if (Array.isArray(parsed)) fields = parsed;

    renderTable($tbody, fields);
    syncJson($tbody, $json);

    $tbody.sortable({
      handle: '.pufsm-drag',
      axis: 'y',
      update: function(){ syncJson($tbody, $json); }
    });

    $table.on('change keyup', 'input, textarea, select', function(){
      var $tr = $(this).closest('tr');
      if ($(this).hasClass('pufsm-type')) toggleOptions($tr);
      syncJson($tbody, $json);
    });

    $table.on('click', '.pufsm-remove', function(){
      if (!confirm(PUFSM_ADMIN && PUFSM_ADMIN.i18n ? PUFSM_ADMIN.i18n.confirm_delete_field : 'حذف شود؟')) return;
      $(this).closest('tr').remove();
      syncJson($tbody, $json);
    });

    $('#pufsm-add-field').on('click', function(){
      var $tr = renderRow({name:'field_'+(Date.now()),label:'فیلد جدید',type:'text',required:false,placeholder:''});
      $tbody.append($tr);
      toggleOptions($tr);
      syncJson($tbody, $json);
    });

    $('#pufsm-apply-template').on('click', function(){
      var tpl = ($('#pufsm-template').val() || '').toString();
      if (!tpl) return;
      var templates = (PUFSM_ADMIN && PUFSM_ADMIN.templates) ? PUFSM_ADMIN.templates : {};
      if (!templates[tpl]) return;
      renderTable($tbody, templates[tpl]);
      syncJson($tbody, $json);
    });
  }

  function initCopyButtons(){
    $(document).on('click', '.pufsm-copy-btn', function(){
      var val = $(this).data('copy') || '';
      // Decode HTML entities if present
      val = val.replace(/&quot;/g, '"');
      if (!val) return;

      var $tmp = $('<textarea style="position:fixed;left:-9999px;top:-9999px;"></textarea>');
      $tmp.val(val);
      $('body').append($tmp);
      $tmp[0].select();
      try { document.execCommand('copy'); } catch(e) {}
      $tmp.remove();

      $(this).text('کپی شد');
      var btn = this;
      setTimeout(function(){ $(btn).text('کپی'); }, 1200);
    });
  }

  $(function(){
    initBuilder();
    initCopyButtons();
  });

})(jQuery);
